/* -*- mode: C; comment-column: 40; fill-column: 79 -*- */
/* $Id: error.c,v 1.169 1999/04/23 15:30:10 dhiller Exp $ */
/* Copyright (C) 1994 - 1998, Hewlett-Packard Company, all rights reserved. */
/* Taken from E1431 library, heavily modified by Eric Backus */

/* This file contains the error handler for the E1432 library. */

#include "sema.h"

/* Warning strings */
static const char * const warningStr[] =
{
    "Warning number out of bounds",	/* shouldn't happen */
    "Lost arm point, not enough tachs",    /*WARN1432_LOST_NOT_ENOUGH_TACHS*/
    "Lost arm point, tach buffer wrapped",    /*WARN1432_LOST_TOO_MANY_TACHS*/
    "Lost arm point, too many points required",    /*WARN1432_LOST_TOO_MANY_POINTS_REQUIRED*/
    "Lost arm point, data shifted out FIFO",    /*WARN1432_LOST_DATA_SHIFTED_OUT_FIFO*/
    "Lost arm point, not enough data in FIFO ",    /*WARN1432_LOST_NOT_ENOUGH_DATA_FIFO*/
    "Lost arm point, RPM too high",    /*WARN1432_LOST_RPM_TOO_HIGH*/
    "Lost arm point, RPM too low",    /*WARN1432_LOST_RPM_TOO_LOW*/
    "Lost arm point, resampling error",    /*WARN1432_LOST_RESAMPLE_ERROR*/
    "Lost arm point, FIFO emptied after stopping",  /*WARN1432_LOST_FIFO_EMPTIED*/
    "RPM is ramping too fast for accurate resampling",  /*WARN1432_RPM_RAMP_TOO_FAST*/
};


/* Error strings */
static const char * const errorStr[] =
{
    "Error number out of bounds",	/* shouldn't happen */

    /* analog */
    "Illegal input or source mode",	/* ERR1432_ILLEGAL_INPUT_MODE */
    "Illegal anti-alias filter state",	/* ERR1432_ILLEGAL_ANTI_ALIAS_MODE */
    "Illegal analog input source",	/* ERR1432_ILLEGAL_INPUT_SOURCE */
    "Illegal input grounding",		/* ERR1432_ILLEGAL_INPUT_GROUNDING */
    "Illegal coupling",			/* ERR1432_ILLEGAL_COUPLING */
    "Illegal ranging down, min. range reached",	/*ERR1432_ILLEGAL_RANGE_DOWN */
    "Illegal ranging up, max. range reached", /* ERR1432_ILLEGAL_RANGE_UP */
    "Illegal range value",		/* ERR1432_RANGE_OUT_OF_LIMITS */
    "Illegal offset value",		/* ERR1432_OFFSET_OUT_OF_LIMITS */
    "Illegal active channel",		/* ERR1432_ILLEGAL_ACTIVE_CHANNEL */

    /* data, getdata */
    "Illegal append status state", /* ERR1432_ILLEGAL_APPEND_STATUS_STATE */
    "Illegal blocksize",		/* ERR1432_ILLEGAL_BLOCKSIZE */
    "Illegal data size",		/* ERR1432_ILLEGAL_DATA_SIZE */
    "Illegal data mode",		/* ERR1432_ILLEGAL_DATA_MODE */
    "Illegal data port",		/* ERR1432_ILLEGAL_DATA_PORT */
    "Illegal channel order",		/* ERR1432_ILLEGAL_CHANNEL_ORDER */
    "Illegal LBUS mode",		/* ERR1432_ILLEGAL_LBUS_MODE */
    "Illegal LBUS state",		/* ERR1432_ILLEGAL_LBUS_STATE */
    "Illegal overlap",			/* ERR1432_ILLEGAL_OVERLAP */

    /* filter */
    "Illegal filter output mode",	/* ERR1432_ILLEGAL_FILTER_PASS_MODE */
    /* ERR1432_ILLEGAL_ANTI_ALIAS_DIG_MODE */
    "Illegal digital anti-alias filter state",
    "Illegal filter decimation bandwidth", /*ERR1432_ILLEGAL_FILTER_BANDWIDTH*/
    "Illegal filter settling time",	/* ERR1432_ILLEGAL_SETTLING_TIME */
    "Illegal zoom state",		/* ERR1432_ILLEGAL_ZOOM_STATE */
    "Illegal center frequency",		/* ERR1432_ILLEGAL_CENTER_FREQ */
    "Illegal ramp state",		/* ERR1432_ILLEGAL_RAMP_STATE */
    /* ERR1432_ILLEGAL_DECIMATION_OVERSAMPLE */
    "Illegal decimation oversample",
    "Illegal span",			/* ERR1432_ILLEGAL_SPAN */
    "FIFO overflow",			/* ERR1432_FIFO_OVERRUN */

    /* timing */
    "Illegal clock source",		/* ERR1432_ILLEGAL_CLOCK_SOURCE */
    "Illegal multi sync mode",		/* ERR1432_ILLEGAL_MULTI_SYNC_MODE */
    "Illegal TTLTRG line specified (0-7)", /* ERR1432_ILLEGAL_TTLTRG_LINE */
    "Illegal sample clock master",	/* ERR1432_ILLEGAL_CLOCK_MASTER */
    "PLL not locked",			/* ERR1432_NO_PLL_LOCK */
    "Illegal group measure state",	/* ERR1432_ILLEGAL_GROUP_MEAS_STATE */
    "Illegal decimation undersamp",	/* ERR1432_ILLEGAL_DECIMATION_UNDERSAMP */
    "Illegal clock frequency",		/* ERR1432_ILLEGAL_CLOCK_FREQ */
    "Illegal clock oversample",		/* ERR1432_ILLEGAL_CLOCK_OVERSAMPLE */

    /* trigger */
    "Illegal triggers per arm",		/* ERR1432_ILLEGAL_TRIGGERS_PER_ARM */
    "Pretrigger delay exceeds size of FIFO",/*ERR1432_PRETRIGGER_GREATER_FIFO*/
    /* ERR1432_ILLEGAL_POST_TRIGGER_DELAY */
    "Illegal post trigger delay out of bounds",
    "Illegal trigger level, out of bounds", /* ERR1432_ILLEGAL_TRIGGER_LEVEL */
    "Illegal trigger level selection",	/* ERR1432_WHICH_TRIGGER_LEVEL */
    "Illegal trigger slope",		/* ERR1432_ILLEGAL_TRIGGER_SLOPE */
    "Illegal trigger mode",		/* ERR1432_ILLEGAL_TRIGGER_MODE */
    "Illegal trigger channel",		/* ERR1432_ILLEGAL_TRIGGER_CHANNEL */
    "Illegal arm state", 		/* ERR1432_ILLEGAL_ARM_STATE */
    "Illegal trigger state", 		/* ERR1432_ILLEGAL_TRIGGER_STATE */
    "Illegal trigger ext",		/* ERR1432_ILLEGAL_TRIGGER_EXT */

    /* group, module, channel */
    "Channel group ID is not valid",	/* ERR1432_NO_GROUP */
    "Group or Channel ID is not valid",	/* ERR1432_NO_ID */
    "Illegal module count",		/* ERR1432_ILLEGAL_MODULE_COUNT */
    "No E1432 module at logical address", /* ERR1432_NO_MOD_AT_LA */
    "Invalid module option",		/* ERR1432_INVALID_MODULE_OPTION */
    "Channel ID is not valid",		/* ERR1432_NO_CHANNEL */
    "Modules in group not in contiguous slots",/*ERR1432_GROUP_NOT_CONTIGUOUS*/
    "Number of modules exceeds limit of 255", /* ERR1432_TOO_MANY_MODULES */
    "Expected group ID",		/* ERR1432_EXPEXT_GROUP */
    "Expected channel ID",		/* ERR1432_EXPEXT_CHANNEL */

    /* io, machine, interrupts */
    "Timeout reading status register",	/* ERR1432_STATUS_READ_TIMEOUT */
    "Bus error",			/* ERR1432_BUS_ERROR */
    "VXI register offset is illegal",	/* ERR1432_ILLEGAL_REGISTER_OFFSET */
    "Illegal interrupt mask",		/* ERR_ILLEGAL_INTR_MASK */
    "Illegal interrupt priority",	/* ERR1432_ILLEGAL_INTR_PRIORITY */
    "Error in system time function, gettimeofday", /* ERR1432_SYSTEM_TIME */
    "Unable to open SICL vxi interface", /* ERR1432_UNABLE_TO_OPEN_SICL */
    "Unable to map address space",	/* ERR1432_UNABLE_TO_MAP_MEMORY */
    "Module error queue corrupt",	/* ERR1432_ERROR_QUEUE_CORRUPT */
    "Unable to install syscall handler", /* ERR1432_IONSRQ */

    /* miscellaneous */
    /* ERR1432_PARAMETER_UNEQUAL */
    "Module parameter not the same across channel group",
    "Unable to malloc memory",		/* ERR1432_MALLOC */
    "Unable to realloc memory",		/* ERR1432_REALLOC */
    "Sigvector failed",			/* ERR1432_SIGVECTOR */
    "Meas state machine not responding", /*ERR1432_MEAS_STATE_NOT_RESPONDING */
    "Auto zero timed out waiting for data", /* ERR1432_AUTO_ZERO_TIME_OUT */
    "E1432 Internal library link list error", /* ERR1432_LIST_ERROR */
    "Auto zero offset DAC failure",	/* ERR1432_AUTO_ZERO_FAILURE */
    "",				/* defunct ERR1432_ILLEGAL_AC_SETTLING */
    "Illegal mode in function call",	/* ERR1432_ILLEGAL_MODE */

    /* eeflash and cal */
    "Illegal cal block specified",	/* ERR1432_BAD_CAL_BLOCK */
    "Posible Hardware error, Flash Rom", /* ERR1432_BAD_FLASH */
    "Rom Vpp Range Error",		/* ERR1432_VPP_ERROR */
    "Rom Command Sequence Error",	/* ERR1432_ROM_COMMAND */
    "Rom Byte Program Error",		/* ERR1432_ROM_BYTE_PROGRAM */
    "Rom Block Erase Error",		/* ERR1432_ROM_BLOCK_ERASE */
    "No Firmware Rev Found in File",	/* ERR1432_NO_FWREV */
    "",
    "",
    "E1432_selftest detects hardware error", /* ERR1432_SELFTEST_FAILURE */

    /* More misc */
    "Data misaligned (need 32bit)",	/* ERR1432_DATA_MISALIGNED */
    "Illegal fifo size",		/* ERR1432_ILLEGAL_FIFO_SIZE */
    "Hardware incapable of request",	/* ERR1432_HARDWARE_INCAPABLE */
    "Firmware error",			/* ERR1432_FIRMWARE_ERROR */
    "",
    "Illegal auto-zero state",		/* ERR1432_ILLEGAL_AUTO_ZERO */
    "Error during block I/O",		/* ERR1432_IO */
    "Error operating on file",		/* ERR1432_FILE */
    "Timeout waiting for sync edge",	/* ERR1432_SYNC_TIMEOUT */
    "Hardware I2C error",		/* ERR1432_I2C */

    /* Source */
    "Illegal amp scale",		/* ERR1432_ILLEGAL_AMP_SCALE */
    "Illegal filter freq",		/* ERR1432_ILLEGAL_FILTER_FREQ */
    "Illegal ramp rate",		/* ERR1432_ILLEGAL_RAMP_RATE */
    "Illegal sine freq",		/* ERR1432_ILLEGAL_SINE_FREQ */
    "Illegal sine phase",		/* ERR1432_ILLEGAL_SINE_PHASE */
    "Illegal source cola",		/* ERR1432_ILLEGAL_SOURCE_COLA */
    "Illegal source sum",		/* ERR1432_ILLEGAL_SOURCE_SUM */
    "Illegal source mode", 		/* ERR1432_ILLEGAL_SOURCE_MODE */
    "Illegal duty cycle", 		/* ERR1432_ILLEGAL_DUTY_CYCLE */
    "Timeout waiting for source ack",	/* ERR1432_ACK_TIMEOUT */

    /* Tach */
    "Illegal tach decimate",		/* ERR1432_ILLEGAL_TACH_DECIMATE */
    "Illegal tach holdoff",		/* ERR1432_ILLEGAL_TACH_HOLDOFF */
    "Illegal tach max time",		/* ERR1432_ILLEGAL_TACH_MAX_TIME */
    "Illegal tach input",		/* ERR1432_ILLEGAL_TACH_INPUT */
    "Illegal tach slope",		/* ERR1432_ILLEGAL_TACH_SLOPE */
    "Illegal tach level",		/* ERR1432_ILLEGAL_TACH_LEVEL */
    "Illegal tach hysteresis",		/* ERR1432_ILLEGAL_TACH_HYST */

    "SICL error",			/* ERR1432_SICL_ERROR */
    "DMA Timeout",			/* ERR1432_DMA_TIMEOUT */
    "DMA Error",			/* ERR1432_DMA_ERROR */

    /* Configuration */
    "SCA Channel count mismatch",	/* ERR1432_CHAN_COUNT_MISMATCH */
    "SCAs not contiguous",		/* ERR1432_SCA_NOT_CONTIG */
    "Source selftest failure",		/* ERR1432_SELFTEST_FAILURE_SOURCE */
    "Source internal error",		/* ERR1432_SOURCE_ERROR */
    "Source overload",			/* ERR1432_SOURCE_OVERLOAD */
    "Source shutdown",			/* ERR1432_SOURCE_SHUTDOWN */
    "Illegal cal dac setting",		/* ERR1432_ILLEGAL_CAL_DAC */
    "Illegal calin setting",		/* ERR1432_ILLEGAL_CALIN */
    "Illegal sumbus setting",		/* ERR1432_ILLEGAL_SUMBUS */
    "Illegal source output",		/* ERR1432_ILLEGAL_SOURCE_OUTPUT */

    /* Dervish */
    "For RPM arm, arm/trigger channel must be the same", /* ERR1432_ILLEGAL_ARM_TRIG_COMBO*/
    "Illegal pre-arm mode",		/* ERR1432_ILLEGAL_PRE_ARM_MODE*/
    "Illegal arm mode",			/* ERR1432_ILLEGAL_ARM_MODE*/
    "Illegal window",			/* ERR1432_ILLEGAL_WINDOW */
    "Illegal Order Track number of orders", /*ERR1432_ILLEGAL_NUMBER_ORDERS*/
    "Illegal pulse per rev", 		/* ERR1432_ILLEGAL_PULSE_PER_REV*/
    "Illegal Order Track delta order", 	/* ERR1432_ILLEGAL_DELTA_ORDER */
    "Illegal Order Track test time", 	/* ERR1432_ILLEGAL_TEST_TIME */
    "Illegal arm time interval", 	/* ERR1432_ILLEGAL_ARM_TIME_INTERVAL */
    "Illegal Order Track blocksize", 	/* ERR1432_ILLEGAL_ORDER_BLOCKSIZE */
    "Illegal Order Track trigger RPM", 	/* ERR1432_ILLEGAL_TRIGGER_RPM */
    "Illegal Order Track channel address", /* ERR1432_ILLEGAL_CHANNEL_ADDRESS*/
    "Illegal Order Track delta max order", /* ERR1432_ILLEGAL_DELTA_MAX_ORDER*/
    "Illegal Order Track order bins", 	/* ERR1432_ILLEGAL_ORDER_BINS */
    "Order Track init param block error", /* ERR1432_PARAM_BLOCK_INIT*/
    "Illegal trigger source", 		/* ERR1432_ILLEGAL_TRIGGER_SOURCE */
    "Illegal RPM interval",		/* ERR1432_ILLEGAL_RPM_INTERVAL*/
    "Illegal RPM low",			/* ERR1432_ILLEGAL_RPM_LOW*/
    "Illegal RPM high",			/* ERR1432_ILLEGAL_RPM_HIGH*/
    "RPM low > RPM high",		/* ERR1432_RPM_LOW_GREATER_HIGH*/
    "Order Track RPM too low for span",	/* ERR1432_RPM_TOO_LOW_FOR_SPAN*/
    "Order Track RPM too high for span", /* ERR1432_RPM_TOO_HIGH_FOR_SPAN*/
    "Illegal trigger source mode", 	/* ERR1432_ILLEGAL_TRIG_SOURCE_MODE*/
    "RPM high not multiple of interval", /* ERR1432_HIGH_NOT_MULT_INTERVAL*/
    "",					/* ERR1432_MALLOC_BLOCK_DATA*/
    "",					/* ERR1432_ILLEGAL_CP_BLOCKSIZE*/
    "Illegal skip N tachs",		/* ERR1432_ILLEGAL_SKIP_N_TACHS*/
    "",					/* ERR1432_ILLEGAL_NUM_DATA_CHANNELS*/
    "",					/* ERR1432_ILLEGAL_DATA_MASK*/
    "Must set arm channel for RPM arming or order track", 	/* ERR1432_NO_ARM_CHANNEL*/
    "Illegal Order Track auto time",	/* ERR1432_ILLEGAL_AUTO_TIME*/
    "Illegal Order Track max order",	/* ERR1432_ILLEGAL_MAX_ORDER  */
    "Illegal Order Track comp power start", /* ERR1432_ILLEGAL_COMP_POWER_START*/
    "Illegal Order Track comp power blocksize",	/* ERR1432_ILLEGAL_COMP_POWER_BLOCKSIZE*/
    "Illegal pulse per rev skip tach", /* ERR1432_ILLEGAL_PPR_SKIP_TACH*/
    "",					/*ERR1432_ILLEGAL_SUB_CHANNEL*/
    "",					/*ERR1432_ILLEGAL_CHAN_ORDER*/
    "Channel is not a tach channel",	/* ERR1432_NOT_TACH_CHANNEL */
    "Illegal master/slave config", 	/* ERR1432_ILLEGAL_MASTER_SLAVE*/
    "Illegal false tach reject", 	/* ERR1432_ILLEGAL_FALSE_TACH_REJ*/
    "Illegal trigger accuracy", 	/* ERR1432_ILLEGAL_TRIGGER_ACCURACY*/
    "Illegal external trigger level", 	/* ERR1432_ILLEGAL_EXT_TRIGGER_LEVEL*/
    "Illegal external trigger chan", 	/* ERR1432_ILLEGAL_EXT_TRIGGER_CHAN*/
    "Illegal external trigger slope", 	/* ERR1432_ILLEGAL_EXT_TRIGGER_SLOPE*/
    "Illegal external trigger hysteresis", /* ERR1432_ILLEGAL_EXT_TRIGGER_HYST*/
    "Tach holdoff & RPM high inconsistant",	/* ERR1432_TACH_HOLDOFF_RPM_HIGH_WRONG*/
    "Illegal RPM smoothing", 		/* ERR1432_ILLEGAL_RPM_SMOOTHING*/
    "Illegal Order Track bandwidth",	/* ERR1432_ILLEGAL_OT_BANDWIDTH*/
    "Illegal number tach chans", 	/* ERR1432_ILLEGAL_NUM_TACH_CHAN*/
    "Illegal Order Track debug flag",	/* ERR1432_ILLEGAL_DEBUG_FLAG*/
    "Illegal Order Track trigger position", /* ERR1432_ILLEGAL_TRIGGER_POSITION*/
    "Illegal trigger master state", /* ERR1432_ILLEGAL_TRIGGER_MASTER*/
    "Illegal Order Track sample mode",	/* ERR1432_ILLEGAL_SAMPLE_MODE*/
    "Order Track RPM step < interval",	/* ERR1432_RPM_STEP_LESS_INTERVAL*/
    "Illegal external trigger number", 	/* ERR1432_ILLEGAL_EXT_TRIGGER_NUM*/
    "Trigger master > num chan", 	/* ERR1432_TRIG_MASTER_GREATER_NUM_CHAN*/
    "Illegal RPM max step",		/* ERR1432_ILLEGAL_RPM_MAX_STEP*/
    "Illegal average mode",		/* ERR1432_ILLEGAL_AVERAGE_MODE*/
    "Illegal average number", 		/* ERR1432_ILLEGAL_AVERAGE_NUMBER*/
    "No tach board found",		/* ERR1432_NO_TACH_BOARD*/
    "Illegal TTL trigger pair", 	/* ERR1432_ILLEGAL_TTL_TRIG_PAIR*/
    "Illegal local bus use", 		/* ERR1432_ILLEGAL_USE_LBUS*/
    "Illegal channel count", 		/* ERR1432_ILLEGAL_CHANNEL_COUNT*/
    "Illegal external trigger channel",	/* ERR1432_ILLEGAL_EXT_TRIGGER_CHANNEL*/
    "Only two tach chans allowed", 	/* ERR1432_ONLY_TWO_TACH_CHAN*/
    "No input SCA found",		/* ERR1432_NO_INPUT_SCA*/
    "RPM high & skip tach inconsistant", /* ERR1432_ILLEGAL_RPM_HIGH_SKIP_TACH*/
    "Can't malloc window buffer", 	/* ERR1432_MALLOC_WINDOW_BUF*/
    "Illegal input SCA",		/* ERR1432_ILLEGAL_OT_INPUT_SCA*/
    "Input data needs DRAM option",	/* ERR1432_NEED_DRAM*/
    "Illegal arm channel",		/* ERR1432_ILLEGAL_ARM_CHANNEL*/
    "Illegal current rpm channel", /* ERR1432_ILLEGAL_CURRENT_RPM_CHANNEL*/
    "Illegal measurement time length",	/* ERR1432_ILLEGAL_MEAS_TIME_LENGTH */
    "Illegal pre arm RPM",		/* ERR1432_ILLEGAL_PRE_ARM_RPM */
    "Illegal calculated data",		/* ERR1432_ILLEGAL_CALC_DATA */
    "",
    "Must set decimation_output to multipass", /* ERR1432_ORDER_MULTIPASS_OFF*/
    "Must set decimation_oversample to oversampled", /* ERR1432_ORDER_OVERSAMPLE_OFF*/
    "Timeout during local bus test",	/* ERR1432_LBUS_TEST_TIMEOUT*/
    "Bad data read during local bus test", /* ERR1432_LBUS_TEST_BAD_DATA*/
    "Illegal local bus test mode",	/* ERR1432_ILLEGAL_LBUS_TEST_MODE*/
    "Order Track lost point",	/* ERR1432_OT_LOST_POINT */
    "Order Track lost point, falling behind",	/* ERR1432_OT_LOST_FALLING_BEHIND*/
    "Order Track FIFO overflow",	/* ERR1432_OT_FIFO_OVERFLOW */
    "Order Track unknown error",	/* ERR1432_OT_UNKNOWN_ERROR*/

    "Bad internal command",		/* ERR1432_BAD_INTERNAL_COMMAND */
    "Calibration failure",		/* ERR1432_CALIBRATION_FAILURE */
    "Calibration not allowed in this state",	/* ERR1432_CAL_DISALLOWED */
    "Source0 data too long",		/* ERR1432_SRC0_XFRDATA_TOO_LONG */
    "Source0 data transfer buffer not ready",	/* ERR1432_SRC0_XFRBUF_NRDY */
    "Illegal source blocksize",		/* ERR1432_ILLEGAL_SRC_BLKSIZE */
    "Illegal source span",		/* ERR1432_ILLEGAL_SRC_SPAN */
    "Illegal source centerfreq",	/* ERR1432_ILLEGAL_SRC_CENTERFREQ */
    "Flash PROM unprogrammed or invalid data",	/* ERR1432_FLASH_INVALID */
    "Pre-arm channel not a tach for RPM pre-arm", /* ERR1432_ILLEGAL_PRE_ARM_CHANNEL*/
    "Illegal srcbuffer mode",		/* ERR1432_ILLEGAL_SRCBUFFER_MODE */
    "Source overread",			/* ERR1432_SOURCE_OVERREAD */
    "Raw tach time buffer overflowed",	/* ERR1432_TACH_BUFFER_OVERFLOW */
    "Timeout waiting for tach time transfer",	/* ERR1432_TACH_XFER_TIMEOUT*/
    "RPM triggered slave rejected trigger",  /* ERR1432_REJECT_SLAVE_TRIGGER*/
    "Illegal coupling frequency",	/* ERR1432_COUPLING_FREQ_OUT_OF_LIMITS */
    "Illegal weighting function",	/* ERR1432_ILLEGAL_WEIGHTING */
    "Illegal input offset",		/* ERR1432_INPUT_OFFSET_OUT_OF_LIMITS */
    "Illegal autozero offset",		/* ERR1432_AUTOZERO_OFFSET_OUT_OF_LIMITS */
    "Duplicate tach time",		/* ERR1432_DUPLICATE_TACH_TIME */
    "Illegal enable",			/* ERR1432_ILLEGAL_ENABLE */
    "Data handshake timeout",		/* ERR1432_DATA_HANDSHAKE_TIMEOUT */
    "Functionality not currently implemented", /* ERR1432_NOT_FUNCTIONAL */
    "SCA hostport hardware failure",	/* ERR1432_SCA_HOSTPORT_FAIL */
    "SCA firmware bootup hardware failure",	/* ERR1432_SCA_BOOT_FAIL */

    /* Selftest failures */
    "Selftest failed - DMA timeout",	/* ERR1432_SELFTEST_DMA_TIMEOUT */
    "Selftest failed - DRAM size",	/* ERR1432_SELFTEST_DRAM_SIZE */
    "Selftest failed - DRAM test",	/* ERR1432_SELFTEST_DRAM */
    "Selftest failed - DRAM full test",	/* ERR1432_SELFTEST_DRAM_FULL */
    "Selftest failed - Y SRAM",		/* ERR1432_SELFTEST_YSRAM */
    "Selftest failed - X IRAM ",	/* ERR1432_SELFTEST_XIRAM */
    "Selftest failed - X SRAM",		/* ERR1432_SELFTEST_XSRAM */
    "Selftest failed - X SRAM2",	/* ERR1432_SELFTEST_XSRAM2 */
    "Selftest failed - XY SRAM",	/* ERR1432_SELFTEST_XYSRAM */
    "Selftest failed - XY SRAM2",	/* ERR1432_SELFTEST_XYSRAM2 */
    "Selftest failed - Tach Rollover",	/* ERR1432_SELFTEST_TACH_ROLLOVER */
    "Selftest failed - Tach Full",	/* ERR1432_SELFTEST_TACH_FULL */
    "Selftest failed - Tach Empty",	/* ERR1432_SELFTEST_TACH_EMPTY */
    "Selftest failed - Tach Count",	/* ERR1432_SELFTEST_TACH_COUNT */

    "Can't enable both frequency and order",	/* ERR1432_ILLEGAL_ENABLE_COMBINATION*/
    "Illegal xfer size",		/* ERR1432_ILLEGAL_XFER_SIZE */
    "Illegal average update number",	/* ERR1432_ILLEGAL_AVG_UPDATE */
    "Illegal average exponential weight", /* ERR1432_ILLEGAL_AVG_WEIGHT */
    "Need newer substrate for E1433",	/* ERR1432_BAD_E1433_SUBSTRATE */
    "E1433 resistor kludge not done",	/* ERR1432_E1433_RESISTOR_FAILURE */
    "Illegal combination: maxOrd > blksize * deltaOrd / 5.12",	/* ERR1432_ILLEGAL_BLOCK_ORDER_COMBO*/
    "Legal order/rpmarm trigger delays: 0, -blksize, -blksize/2",/* ERR1432_ILLEGAL_ORDER_TRIGGER_DELAY*/
    "Averaging with RPM triggering not allowed",/* ERR1432_NO_AVG_WITH_RPM_TRIG*/
    "Illegal range_charge setting",	/* ERR1432_ILLEGAL_RANGE_CHARGE */
    "Illegal range_mike setting",	/* ERR1432_ILLEGAL_RANGE_MIKE */
    "Max of 8 chans on two SCAs at top span",	/* ERR1432_ILLEGAL_ORDER_CHANNEL_COMBO*/
    "SCA bus error",			/* ERR1432_SCA_BUS_ERROR */
    "SCA firmware error",		/* ERR1432_SCA_FIRMWARE_ERROR */
    "SCA ADC data overrun",		/* ERR1432_SCA_ADC_DATA_OVERRUN */
    "SCA DSP data overrun",		/* ERR1432_SCA_DSP_DATA_OVERRUN */
    "RMS span not 2^n multiple of normal span", /* ERR1432_RMS_SPAN_MISMATCH */
    "Illegal RMS span",                 /* ERR1432_ILLEGAL_RMS_SPAN */
    "Illegal RMS decay time",           /* ERR1432_ILLEGAL_RMS_DECAY_TIME */
    "RMS span changed since RMS decay time last set",  /* ERR1432_RMS_DECAY_TIME_MISMATCH */
    "Illegal Peak hold time",           /* ERR1432_ILLEGAL_PEAK_HOLD_TIME */
    "Insufficient buffer space available",	/* ERR1432_BUFFER_TOO_SMALL */
/*** REQUEST (a tag) ***/
    "Prior requested spectrum in progress",  /* ERR1432_PRIOR_REQUESTED_SPECTRUM */
    "No spectrum request or spectrum not ready",  /* ERR1432_NO_SPECTRUM_READY */
    "File is wrong type or contains bad data",  /* ERR1432_SRC_ROMFILE_CHECKCODE */
    "File size is not correct",  /* ERR1432_SRC_ROMFILE_SIZE */
    "File is incomplete or contains bad data",  /* ERR1432_SRC_ROMFILE_TRAILER */
    "Illegal Peak mode",		/* ERR1432_ILLEGAL_PEAK_MODE */
    "Illegal RMS mode",			/* ERR1432_ILLEGAL_RMS_MODE */
    "Illegal current value type",	/* ERR1432_ILLEGAL_CURRENT_VALUE */
    "Illegal RMS average time",		/* ERR1432_ILLEGAL_RMS_AVG_TIME */
    "Y memory allocation error",	/* ERR1432_Y_MEMORY_ALLOCATION */
    "Input weighting unavailable with current clock frequency",
					/* ERR1432_WEIGHTING_VS_CLOCK_F */
    "Clock frequency too high for decimated span",
					/* ERR1432_DECIMATION_VS_CLOCK_F */
    "Clock frequency too high for Peak/RMS computation",
					/* ERR1432_PEAK_RMS_VS_CLOCK_F */
    "Auto-range failed to converge",	/* ERR1432_AUTO_RANGE_FAILURE */
    "Illegal Peak decay time",		/* ERR1432_ILLEGAL_PEAK_DECAY_TIME */
    "Illegal auto range mode",		/* ERR1432_ILLEGAL_AUTO_RANGE_MODE */
    /* diagnostics failures */
    "Undiagnosed failure",		/* ERR1432_DIAG_ERR_UNDIAGNOSED */
    "Substrate board failed diagnostics", /* ERR1432_DIAG_ERR_SUBST */
    "DRAM SIM failed diagnostics",	/* ERR1432_DIAG_ERR_DRAM */
    "Tach board failed diagnostics",	/* ERR1432_DIAG_ERR_TACH */
    "Rear Source board failed diagnostics",   /* ERR1432_DIAG_ERR_SOURCE_FIT */
    "Top source board failed diagnostics",    /* ERR1432_DIAG_ERR_SOURCE_SCA0 */
    "Bottom source board failed diagnostics", /* ERR1432_DIAG_ERR_SOURCE_SCA1 */
    "Input SCA 1 failed diagnostics",	/* ERR1432_DIAG_ERR_INPUT_SCA0 */
    "Input SCA 2 failed diagnostics",	/* ERR1432_DIAG_ERR_INPUT_SCA1 */
    "Input SCA 3 failed diagnostics",	/* ERR1432_DIAG_ERR_INPUT_SCA2 */
    "Input SCA 4 failed diagnostics",	/* ERR1432_DIAG_ERR_INPUT_SCA3 */

    "Clock frequency too high for auto-range",
					     /* ERR1432_AUTO_RANGE_VS_CLOCK_F */
    "Source realtime restriction (see e1432_set_source_mode)",
					  /* ERR1432_SRC_REALTIME_RESTRICTION */
    "Illegal Octave mode",		/* ERR1432_ILLEGAL_OCTAVE_MODE */
    "Illegal Octave average mode",	/* ERR1432_ILLEGAL_OCTAVE_AVG_MODE */
    "Illegal Octave hold mode",		/* ERR1432_ILLEGAL_OCTAVE_HOLD_MODE */
    "Illegal Octave start frequency",	/* ERR1432_ILLEGAL_OCTAVE_START_FREQ */
    "Illegal Octave stop frequency",	/* ERR1432_ILLEGAL_OCTAVE_STOP_FREQ */
    "Illegal Octave integration time",	/* ERR1432_ILLEGAL_OCTAVE_INT_TIME */
    "Illegal Octave time constant",	/* ERR1432_ILLEGAL_OCTAVE_TIME_CONST */
    "Illegal Octave time step",		/* ERR1432_ILLEGAL_OCTAVE_TIME_STEP */
    "Illegal Octave clock frequency",	/* ERR1432_ILLEGAL_OCTAVE_CLOCK_FREQ */
    "Illegal Octave control parm",	/* ERR1432_ILLEGAL_OCTAVE_CTL */
    "Illegal user decimation",		/* ERR1432_ILLEGAL_USER_DECIMATION */
    "Illegal user frames_per_block", /* ERR1432_ILLEGAL_USER_FRAMES_PER_BLOCK */
    "Illegal get_current_data channel",
				     /* ERR1432_ILLEGAL_GET_CURRENT_DATA_CHAN */
    "Illegal get_current_data type", /* ERR1432_ILLEGAL_GET_CURRENT_DATA_TYPE */
    "Illegal Octave/user blocksize",	/* ERR1432_ILLEGAL_USER_BLOCKSIZE */
    "Aux FIFO overrun",			/* ERR1432_AF_OVERRUN */
    "Required option not installed",	/* ERR1432_OPTION_NOT_INSTALLED */
    "Invalid option string",		/* ERR1432_INVALID_OPTION_STRING */
    "Illegal tach irq number",		/* ERR1432_ILLEGAL_TACH_IRQ_NUMBER */
    "Data is unavailable",		/* ERR1432_DATA_UNAVAILABLE */
    "Illegal get_current_data data size",
				/* ERR1432_ILLEGAL_GET_CURRENT_DATA_SIZE */
    "Clock frequency too high for 32-bit data size", /* ERR1432_DATA_SIZE_VS_CLOCK_FREQ */
    "Clock frequency too high for Octave", /* ERR1432_OCTAVE_VS_CLOCK_FREQ */
    "Incorrect module type for Octave",	/* ERR1432_OCTAVE_VS_MODULE_TYPE */
    "Illegal source buffer size",	/* ERR1432_ILLEGAL_SRCBUFFER_SIZE */
    "Illegal Octave start/stop frequency combination",
					/* ERR1432_ILLEGAL_OCTAVE_FREQS */
    "Illegal Octave measurement value",	/* ERR1432_ILLEGAL_OCTAVE_MEAS */
    "Illegal source buffer init mode",	/* ERR1432_ILLEGAL_SRCBUFFER_INIT */
    "Decimation unavailable with Octave",/* ERR1432_ILLEGAL_OCTAVE_DECIMATION */
    "Internal error in host interface library", /* ERR1432_INTERNAL_LIBRARY_ERROR */
    "Illegal source data mode",		/* ERR1432_ILLEGAL_SRC_DATA_MODE */
    "Attempt to use DRAM for both input and source channels", /*ERR1432_DRAM_CONFLICT */
    "Max of 4 chans on two SCAs at top span",	/* ERR1432_ILLEGAL_ORDER_CHANNEL_COMBO33*/
    "Hostlib/sema.bin version mismatch",/* ERR1432_SEMA_BIN_MISMATCH */
    "Serial number change not allowed",	/* ERR1432_SERIAL_NUM_ZAP_NOT_ALLOWED */
    "This firmware requires E1432 option 204", /* ERR1432_REQUIRE_VIBRATO_B */
    "Only one arm channel supported on multi-module systems", /* ERR1432_ILLEGAL_ARM_CHAN_NUM */
    "Need running measurement to request a spectrum", /* ERR1432_REQUEST_MEAS_RUNNING */
    "Span change not allowed during request spectrum", /* ERR1432_REQUEST_SPAN_CHANGE */
    "Request spectrum not allowed when zooming", /* ERR1432_REQUEST_ZOOM */
    "PEAK span not 2^n multiple of normal span", /*ERR1432_PEAK_SPAN_MISMATCH */
    "Illegal PEAK span",                 /* ERR1432_ILLEGAL_PEAK_SPAN */
    "HPF span not 2^n multiple of normal span", /*ERR1432_HPF_SPAN_MISMATCH */
    "Illegal HPF span",                 /* ERR1432_ILLEGAL_HPF_SPAN */
    "Peak span changed since peak hold time last set", /* ERR1432_PEAK_HOLD_TIME_MISMATCH */
    "Incorrect SCA DSP downloadable",	/* ERR1432_BAD_SCA_DOWNLOADABLE */
    "Read CVT data from one module at a time", /* ERR1432_REQUIRE_ONE_MOD */
    "Unable to lock or unlock the VXI session", /* ERR1432_LOCK_FAILED */
    "Invalid SCA EEPROM data",		/* ERR1432_SCA_EEPROM_INVALID */
};


/* error_info_buf[] saves the most recent i1432_error_info string */
#ifndef HPVXI_DOWNLOAD
#define ERROR_INFO_BUF_SIZE 80
static char error_info_buf[ERROR_INFO_BUF_SIZE] = { 0 }; 
#endif /* HPVXI_DOWNLOAD */

char   * EXPORT
e1432_get_warning_string(SHORTSIZ16 warning)
{
    /* Check bounds, set to unknown if out of range.  This assumes
       positive warning numbers */
    if (warning < WARN1432_NO_WARNING || warning > WARN1432_LAST_WARNING_NUMBER)
	warning = WARN1432_NO_WARNING;

    return (char *) warningStr[warning];
}

char   * EXPORT
e1432_get_error_string(void)
{
    return i1432_error_str;
}

char   *
i1432_get_error_string(SHORTSIZ16 error)
{
    int     index;

    /* Check bounds, set to unknown if out of range.  This assumes
       negative error numbers */
    if (error > ERR1432_ERROR_OFFSET ||
	error < ERR1432_LAST_ERROR_NUMBER)
	error = ERR1432_ERROR_OFFSET;

    /* Get index into errorStr array.  This assumes negative error
       numbers. */
    index = ERR1432_ERROR_OFFSET - error;

    return (char *) errorStr[index];
}

/*
 *********************************************************************
 e1432_fill_error_string() performs the same function as
 e1432_get_error_string(), except it copies the string rather than
 returning a char*.  This form is Visual Basic friendly because
 VB strings are not the same as C strings.

 Example VB code:
    Dim actual_len As Integer
    Dim error_string As String * 128   'fixed string for DLL parm
    Dim error_str as String            'normal VB string
    er = e1432_fill_error_string(ByVal error_string, 128, actual_len)
    error_str = Left$(s, actualLen)
 *********************************************************************
 */
#ifndef HPVXI_DOWNLOAD
SHORTSIZ16 EXPORT
e1432_fill_error_string(char *error_string,
			SHORTSIZ16 max_len, SHORTSIZ16 *actual_len)
{
    error_string[0] = 0;
    (void) strncat(error_string, i1432_error_str, max_len - 1);
    *actual_len = strlen(error_string);

    /* append extra error_info */
    if (strlen(error_info_buf) && 
    	((int)strlen(error_info_buf)+ 3 + *actual_len < max_len - 1)) {
    	(void) strcat(error_string, "\n(" );      
    	(void) strcat(error_string, error_info_buf );
    	(void) strcat(error_string, ")" );      
    	*actual_len = strlen(error_string);
    }
    	
    return 0;
}
#endif

static const char unknownerror[] = "unknown error";

/*
 *********************************************************************
 The function i1432_print_error() is called whenever an unrecoverable
 error is detected by a E1432 library function.  Returns the error
 number, or -1 if illegal error number.
 *********************************************************************
 */
SHORTSIZ16
i1432_la_print_error(SHORTSIZ16 la, SHORTSIZ16 error)
{
    char   *error_str;

#ifdef HPVXI_DOWNLOAD
    char    cl[50];
#endif

    /* get the string descriptor for the error */
    error_str = i1432_get_error_string(error);
    if (!error_str)
	error_str = (char *) unknownerror;

    /* point global error pointer to this error */
    i1432_error_str = error_str;

    /* return now if error printing not enabled */
    if (!i1432_print_errors)
    {
	/* Make a persistent copy of i1432_error_info */
	if (i1432_error_info != NULL)
	{
#ifndef	HPVXI_DOWNLOAD
	    (void) strncpy(error_info_buf, i1432_error_info,
			   ERROR_INFO_BUF_SIZE);
	    error_info_buf[ERROR_INFO_BUF_SIZE - 1] = 0;
#endif
	    i1432_error_info = NULL;	/* clear extra error info */
	}
#ifndef	HPVXI_DOWNLOAD
	else
	    error_info_buf[0] = 0;	/* no extra error info */
#endif
	return error;
    }

    /* print the string */
#ifdef	ALLOW_PRINTF
#if defined(E1485_SOURCE) || defined(_WINDLL) || defined(_WIN32) \
 || defined(HPVXI_DOWNLOAD)
    if (la >= 0)
	PRINTF(("E1432 library detected error %d at la %d\n", error, la));
    else
	PRINTF(("E1432 library detected error %d\n", error));
#else
    if (la >= 0)
	(void) fprintf(stderr, "E1432 library detected error %d at la %d\n",
		       error, la);
    else
	(void) fprintf(stderr, "E1432 library detected error %d\n",
		       error);
#endif
#endif
#ifdef HPVXI_DOWNLOAD
    if (la >= 0)
	(void) sprintf(cl, "E1432 library detected error %d at la %d",
		       error, la);
    else
	(void) sprintf(cl, "E1432 library detected error %d", error);
    os_display_message(5, cl);
    os_pause_p(10);
    os_display_message(5, error_str);
    os_pause_p(10);
#endif /* HPVXI_DOWNLOAD */

    if (i1432_error_info == NULL)
    {
#ifdef	ALLOW_PRINTF
#if defined(E1485_SOURCE) || defined(_WINDLL) || defined(_WIN32) \
 || defined(HPVXI_DOWNLOAD)
	PRINTF(("\"%s\"\n", error_str));
#else
	(void) fprintf(stderr, "\"%s\"\n", error_str);
#endif
#endif
#ifndef HPVXI_DOWNLOAD
    	error_info_buf[0] = 0;		/* no extra error info */
#endif
    }
    else
    {
#ifdef	ALLOW_PRINTF
#if defined(E1485_SOURCE) || defined(_WINDLL) || defined(_WIN32) \
 || defined(HPVXI_DOWNLOAD)
	PRINTF(("\"%s\" (%s)\n", error_str, i1432_error_info));
#else
	(void) fprintf(stderr, "\"%s\" (%s)\n", error_str, i1432_error_info);
#endif
#endif
#ifdef	HPVXI_DOWNLOAD
	os_display_message(5, i1432_error_info);
	os_pause_p(10);
#endif
#ifndef HPVXI_DOWNLOAD
	/* make a persistent copy of i1432_error_info */
    	(void) strncpy(error_info_buf, i1432_error_info, ERROR_INFO_BUF_SIZE);
	error_info_buf[ERROR_INFO_BUF_SIZE - 1] = 0;
#endif
	i1432_error_info = NULL;	/* clear extra error info */
    }
    return error;
}

SHORTSIZ16
i1432_id_print_error(E1432ID hw, SHORTSIZ16 ID, SHORTSIZ16 error)
{
    E1432_MODULE_LIST_NODE *mn;

    if (ID > 0 && i1432_get_module_from_chan(hw, ID, &mn) >= 0)
	return i1432_la_print_error(mn->la, error);
    else
	return i1432_la_print_error(-1, error);
}

SHORTSIZ16
i1432_print_error(SHORTSIZ16 error)
{
    return i1432_la_print_error(-1, error);
}

/*
 *********************************************************************
 Sets the print flag to <state> (i.e. 0 or 1) Always returns 0.
 *********************************************************************
 */
SHORTSIZ16 EXPORT
e1432_print_errors(SHORTSIZ16 state)
{
    TRACE_PRINTF(0, ("e1432_print_errors(%d)\n", state));
    i1432_print_errors = state;
    return 0;
}

/*
 *********************************************************************
 Reads the measurement warnings
 *********************************************************************
 */
SHORTSIZ16 EXPORT
e1432_get_meas_warning(E1432ID hw, SHORTSIZ16 ID, SHORTSIZ16 *warning,
		       unsigned long size, unsigned long *actualCount)
{
    SHORTSIZ16 error, chan, done, mod, status, error2; 
    unsigned long count;
    LONGSIZ32 temp;
    E1432_GROUP_LIST_NODE *gn;

    gn = i1432_get_group_node(hw, ID);
    if(!gn) return i1432_print_error(ERR1432_NO_GROUP);
    
    count = 0;
    done = 0;

    /* get all the warnings from all modules in the group */
    while(!done && count < size)
    {
	done = 1;
        for (mod = 0; mod < gn->modcount; mod++)
        {
	    error = i1432_direct_read_register(gn->modlist[mod],
					E1432_IRQ_STATUS2_REG, &status);
	    if (error < 0)
		return error;

	    if(!(status & E1432_IRQ_MEAS_WARNING))
		continue;

            chan = i1432_get_chan_from_module(gn->modlist[mod]);

	    error = i1432_introff();
	    if (error)
		return error;

            error = i1432_write_cmd0(hw, chan, E1432_CMD_READ_WARNING);
            if (error >= 0)
		error = i1432_read_resp(hw, chan, &temp);

	    error2 = i1432_intron();
	    if (error2)
		return error2;

            if (error < 0)
               return error;

            if(temp != 0)
	    {
	        *warning++ = (SHORTSIZ16) temp;
		count++;
		done = 0;
	    }
	    
	    if(count == size) 
		break;

        }
    }

    *actualCount = count;
    return 0;
}

/* copy a string into error_info_buf, this is used by VISA errors */
#ifdef	HAVE_VTL
#ifndef HPVXI_DOWNLOAD
void i1432_set_error_info(char *es)
{
    (void) strncpy(error_info_buf,es,ERROR_INFO_BUF_SIZE);
    error_info_buf[ERROR_INFO_BUF_SIZE - 1] = 0;
    return;
}
#endif /* HPVXI_DOWNLOAD */
#endif	/* HAVE_VTL */
